package unificationSolver;


/**
 * @author Bill
 *
 * The strategy implied by AbstractSolutionNode is straightforward. It saves the substition set
 * with which it was created as the context of its subtree. It then creates new substitution sets
 * to complete the solution. This allows backtracking to the original substitution set (original state).
 * 
 * It also keeps pointers into the rule set, remembering the last rule tried. When these pointers "run off the end"
 * of the rule set, the node has finished its search, causing backtrack to the parent.
 * 
 * Essentially, solution nodes are analogous to stack frames in the search. 
 */
public abstract class AbstractSolutionNode {
	
	private RuleSet rules;
	private Rule currentRule = null;
	private Goal goal= null;
	public static int nodesCreated = 0;
	
	// saving the parent solution allows backtracking to the original state
	private SubstitutionSet parentSolution;
	
	//These variables allow the solution node to iterate over the rule set.
	private int ruleNumber = 0;
	
	public AbstractSolutionNode(Goal goal, RuleSet rules, SubstitutionSet parentSolution)
	{
		this.rules = rules;
		this.parentSolution = parentSolution;
		this.goal = goal;
	}
	
	public abstract SubstitutionSet nextSolution()  throws CloneNotSupportedException ;

	protected void reset(SubstitutionSet newParentSolution){
		parentSolution = newParentSolution;
		ruleNumber = 0;
	}

	public Rule nextRule() throws CloneNotSupportedException{
		if(hasNextRule())
			currentRule = rules.getRuleStandardizedApart(ruleNumber++);
		else
			currentRule = null;
		return currentRule;
	}
	
	public boolean hasNextRule(){
		return ruleNumber < rules.getRuleCount();
	}
	
	public SubstitutionSet getParentSolution(){
		return parentSolution;
	}
	
	public RuleSet getRuleSet(){
		return rules;
	}
	
	public Rule getCurrentRule(){
		return currentRule;
	}
	
	public Goal getGoal(){
		return goal;
	}

}
